//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12
import QtQuick.Layouts 1.12

import wesual.Create 1.0
import wesual.Stage  1.0
import wesual.Ui     1.0

import wesual.Ui.Private 1.0

import wesual.data.Core 1.0

import wesual.module.Dashboard 1.0

UiPopupButton {
    id : picker

    property SizeInfo activeSize : Size.defaultSize()

    property Component headerDelegate

    signal sizePicked(SizeInfo size)

    function transposeSize() {
        const transposed = Size.sizeInfo(
                             activeSize.height, activeSize.width);
        sizePicked(transposed);
    }

    buttonDelegate : SizeIcon {
        size : picker.activeSize
        highlight : picker.highlight
    }

    popupDelegate : UiTooltipAgent {
        id : popup

        readonly property SizeDatabase db : Creator.context.sizeDatabase

        implicitWidth : Math.max(470, content.implicitWidth + 20)
        implicitHeight : Math.min(400, content.implicitHeight + 20)
        opacity : 0

        states : [
            State {
                name : "visible"
                when : picker.down

                PropertyChanges {
                    target  : popup
                    opacity : 1
                }
            }
        ]

        transitions : Transition {
            NumberAnimation {
                property : "opacity"
                duration : 150
            }
        }

        Rectangle {
            anchors.fill : parent
            border {
                width : 1
                color : "#CCC"
            }

            Rectangle {
                anchors {
                    top : parent.top
                    bottom : parent.bottom
                    left : parent.left
                    margins : 1
                }
                color : "#F6F6F8"
                width : sizePanel.width + 16
            }
        }

        RowLayout {
            id : content

            anchors.fill : parent
            anchors.margins : 10
            anchors.rightMargin : 4

            spacing : 12

            ColumnLayout {
                id : sizePanel

                readonly property real preferredWidth :
                    Math.max(implicitWidth, 200)

                spacing : 4

                Layout.fillWidth : false
                Layout.preferredWidth : preferredWidth
                Layout.alignment : Qt.AlignLeft | Qt.AlignTop

                Loader {
                    sourceComponent : picker.headerDelegate
                    visible : picker.headerDelegate !== null

                    Layout.fillWidth : true
                    Layout.bottomMargin : 2
                }

                UiSectionHeader {
                    text : qsTrId("1ece5d71125bd900")
                }

                Loader {
                    id : sizeView

                    property point editPos

                    function editSize(pos) {
                        editPos = pos || Qt.point(0, 0);
                        sourceComponent = sizeEditorComponent;
                    }

                    sourceComponent : sizeLabelComponent
                    onItemChanged : {
                        if (item.editAt) {
                            item.editAt(editPos);
                        }
                    }

                    Layout.fillWidth : true
                }

                Component {
                    id : sizeLabelComponent

                    RowLayout {
                        Text {
                            font : UiFonts.getFont(UiFonts.Light, 18)
                            text : {
                                let txt = picker.activeSize.resolution;

                                const rn = picker.activeSize.resolutionName;
                                if (rn.length > 0) {
                                    txt += " (" + rn + ")"
                                }

                                return txt;
                            }

                            MouseArea {
                                anchors.fill : parent
                                cursorShape : Qt.IBeamCursor

                                onClicked : sizeView.editSize(
                                                Qt.point(mouse.x, mouse.y))
                            }
                        }

                        UiToolButton {
                            iconName : "edit"
                            iconSize : UiIcon.Icon_16x16

                            onClicked : sizeView.editSize()
                        }
                    }
                }

                Component {
                    id : sizeEditorComponent

                    SizeEditor {
                        font : UiFonts.getFont(UiFonts.Light, 18)
                        size : picker.activeSize

                        onEditFinished : {
                            if (size) {
                                picker.sizePicked(size);
                            }
                            sizeView.sourceComponent = sizeLabelComponent;
                        }
                    }
                }

                AspectRatioToggle {
                    id : toggle

                    size : picker.activeSize

                    onTransposeOrientation : {
                        popup.forceActiveFocus();
                        picker.transposeSize();
                    }

                    Layout.bottomMargin : 6
                }

                UiCommandLink {
                    text : qsTrId("fa6a1e8df19ac6e6")

                    onClicked :{
                        popup.forceActiveFocus();
                        sizeList.createCustomSize(picker.activeSize);
                    }
                }
            }

            ColumnLayout {
                spacing : 4

                UiSectionHeader {
                    text : qsTrId("ff001e73c2cf7413")

                    Layout.leftMargin : 4
                }

                SizeList {
                    id : sizeList

                    activeSize : picker.activeSize
                    sizeDatabase : popup.db

                    onSizePicked : picker.sizePicked(size)

                    Layout.fillWidth : true
                    Layout.fillHeight : true
                    Layout.preferredHeight : preferredHeight
                }
            }
        }
    }
}
